/*
Copyright 2008-2009 Elöd Egyed-Zsigmond, Cyril Laitang
Copyright 2009-2011 Samuel Gesche

This file is part of IPRI News Analyzer.

IPRI News Analyzer is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

IPRI News Analyzer is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with IPRI News Analyzer.  If not, see <http://www.gnu.org/licenses/>.
*/

package proc.thema;

import data.base.Database;
import data.base.NoBaseException;
import data.base.connectors.LocalSemanticDatabase;

import data.structures.classification.*;

import data.structures.corpus.CorpusGraph;
import data.structures.corpus.CorpusGraphLemmaItem;
import data.structures.thema.ThemaGraph;
import data.structures.tagging.GraphLemmaItem;
import data.structures.tagging.LemmaItem;

import proc.text.Horloge;

import java.util.Vector;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;

public class Experience implements Runnable {

    public final static int VECTEUR_PRODUIT_SCALAIRE = 1;
    public final static int VECTEUR_DISTANCE_EUCLIDIENNE = 2;
    public final static int VECTEUR_DISTANCE_DE_MANHATTAN = 3;

    public Experience() {
    }

    private Database base;
    private String titre;
    private String desc;
    private CorpusGraph corpus;
    private ThemaGraph themas;
    private int[] themaTrouvees;
    private String statut;
    private boolean enrichThemaBase = false;
    private boolean enrichThemaSpe = false;
    private boolean enrichThemaGen = false;
    private boolean enrichThemaCat = false;
    private boolean enrichThemaSem = false;
    private boolean enrichArticleBase = false;
    private boolean enrichArticleGen = false;
    private boolean enrichArticleCat = false;
    private boolean enrichArticleSem = false;
    private boolean normalisationTaille = false;
    private Map<Operation, Double> poidsOrigines = new HashMap<Operation, Double>();
    private Map<String, Double> poidsCategories = new HashMap<String, Double>();
    private double poidsCategoriesNonSpecifie = 1.0;
    private int algoCalcul = VECTEUR_PRODUIT_SCALAIRE;

    Corpus cor = new Corpus();
    Taxonomie tax = new Taxonomie();

    public void setBase(Database base){
        this.base = base;
    }

    public String getDesc() {
        return desc;
    }

    public void setDesc(String desc) {
        this.desc = desc;
    }

    public String getTitre() {
        return titre;
    }

    public void setTitre(String titre) {
        this.titre = titre;
    }

    public boolean isEnrichThemaBase() {
        return enrichThemaBase;
    }

    public void setEnrichThemaBase(boolean enrichThemaBase) {
        this.enrichThemaBase = enrichThemaBase;
    }

    public boolean isEnrichThemaSpe() {
        return enrichThemaSpe;
    }

    public boolean isEnrichArticleCat() {
        return enrichArticleCat;
    }

    public void setEnrichArticleCat(boolean enrichArticleCat) {
        this.enrichArticleCat = enrichArticleCat;
    }

    public boolean isEnrichThemaCat() {
        return enrichThemaCat;
    }

    public void setEnrichThemaCat(boolean enrichThemaCat) {
        this.enrichThemaCat = enrichThemaCat;
    }

    public void setEnrichThemaSpe(boolean enrichThemaSpe) {
        this.enrichThemaSpe = enrichThemaSpe;
    }

    public boolean isEnrichThemaGen() {
        return enrichThemaGen;
    }

    public void setEnrichThemaGen(boolean enrichThemaGen) {
        this.enrichThemaGen = enrichThemaGen;
    }

    public boolean isEnrichArticleBase() {
        return enrichArticleBase;
    }

    public void setEnrichArticleBase(boolean enrichArticleBase) {
        this.enrichArticleBase = enrichArticleBase;
    }

    public boolean isEnrichArticleGen() {
        return enrichArticleGen;
    }

    public void setEnrichArticleGen(boolean enrichArticleGen) {
        this.enrichArticleGen = enrichArticleGen;
    }

    public void setPoidsOrigine(Operation origine, double poids){
        poidsOrigines.put(origine, new Double(poids));
    }

    public void annulePoidsCategories(){
        poidsCategoriesNonSpecifie = 0.0;
    }

    public void setPoidsCategorie(String categorie, double poids){
        poidsCategories.put(categorie, new Double(poids));
    }

    public void setCorpus(CorpusGraph corpus) {
        this.corpus = corpus;
        themaTrouvees = new int[corpus.getItems().length];
    }

    public void setThemas(ThemaGraph themas) {
        this.themas = themas;
    }

    public boolean isEnrichArticleSem() {
        return enrichArticleSem;
    }

    public void setEnrichArticleSem(boolean enrichArticleSem) {
        this.enrichArticleSem = enrichArticleSem;
    }

    public boolean isEnrichThemaSem() {
        return enrichThemaSem;
    }

    public void setEnrichThemaSem(boolean enrichThemaSem) {
        this.enrichThemaSem = enrichThemaSem;
    }

    public boolean isNormalisationTaille(){
        return normalisationTaille;
    }

    public void setNormalisationTaille(boolean normalisationTaille){
        this.normalisationTaille = normalisationTaille;
    }

    public int getAlgoCalcul() {
        return algoCalcul;
    }

    public void setAlgoCalcul(int algoCalcul) {
        this.algoCalcul = algoCalcul;
    }

    public int getNbLemmesCoteArticle() {
        if(corpus==null){
            return 0;
        }

        int nb = 0;
        for (int i = 0; i < cor.getArticles().length; i++) {
            nb += cor.getArticles()[i].getAllLemmes().length;
        }
        return nb;
    }

    public int getNbLemmesCoteThematique() {
        if(themas==null){
            return 0;
        }

        int nb = 0;
        for (int i = 0; i < tax.getThematiques().length; i++) {
            nb += tax.getThematiques()[i].getAllLemmes().length;
        }
        return nb;
    }

    private void creeNuageCorpus() throws NoBaseException {
        // * Récupération des articles
        System.out.println(Horloge.getHMS()+"  Articles");

        // Récupération de base : les lemmes des articles, tout simplement
        GraphLemmaItem[] lCor = corpus.getItems();
        for(int i=0; i<lCor.length; i++){
            statut = "Enrichissement article de base ("+(i+1)+"/"+lCor.length+");";
            cor.addArticle(lCor[i]);
        }

        if(isEnrichArticleSem()){
            for(int i=0; i<cor.getArticles().length; i++){
                statut = "Enrichissement article par champ sémantique ("+(i+1)+"/"+cor.getArticles().length+")";
                System.out.println(Horloge.getHMS()+"   Article "+(i+1)+"/"+cor.getArticles().length+" : "+cor.getArticles()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<cor.getArticles()[i].getAllLemmes().length; j++){
                    //if(j==0) System.out.println("    Test de l'enrichissement par généralisation");
                    LemmePondere lemme = cor.getArticles()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+cor.getArticles()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    //if(j==0) System.out.println("    Premier lemme : "+lemme.getLemme()+" ("+lemme.getCategorie()+")");
                    // enrichissement sémantique de 'lemme'.
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getChampSemantique(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        //if(j==0) System.out.println("    Nombre de parents dans l'ontologie : "+enrichissement.length);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.CHAMP_SEMANTIQUE);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    cor.getArticles()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }

        if(isEnrichArticleCat()){
            for(int i=0; i<cor.getArticles().length; i++){
                statut = "Enrichissement article par catégorisation ("+(i+1)+"/"+cor.getArticles().length+")";
                System.out.println(Horloge.getHMS()+"   Article "+(i+1)+"/"+cor.getArticles().length+" : "+cor.getArticles()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<cor.getArticles()[i].getAllLemmes().length; j++){
                    //if(j==0) System.out.println("    Test de l'enrichissement par généralisation");
                    LemmePondere lemme = cor.getArticles()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+cor.getArticles()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    //if(j==0) System.out.println("    Premier lemme : "+lemme.getLemme()+" ("+lemme.getCategorie()+")");
                    // enrichissement sémantique de 'lemme'.
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getGeneralisationWiki(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        //if(j==0) System.out.println("    Nombre de parents dans l'ontologie : "+enrichissement.length);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.CATEGORISATION);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    cor.getArticles()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }

        if(isEnrichArticleGen()){
            for(int i=0; i<cor.getArticles().length; i++){
                statut = "Enrichissement article par généralisation ("+(i+1)+"/"+cor.getArticles().length+")";
                System.out.println(Horloge.getHMS()+"   Article "+(i+1)+"/"+cor.getArticles().length+" : "+cor.getArticles()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<cor.getArticles()[i].getAllLemmes().length; j++){
                    //if(j==0) System.out.println("    Test de l'enrichissement par généralisation");
                    LemmePondere lemme = cor.getArticles()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+cor.getArticles()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    //if(j==0) System.out.println("    Premier lemme : "+lemme.getLemme()+" ("+lemme.getCategorie()+")");
                    // enrichissement sémantique de 'lemme'.
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getGeneralisationOnto(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        //if(j==0) System.out.println("    Nombre de parents dans l'ontologie : "+enrichissement.length);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.GENERALISATION);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    cor.getArticles()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }

    }

    private void creeNuageTaxonomie() throws NoBaseException {
        // * Récupération des thématiques
        System.out.println(Horloge.getHMS()+"  Thématiques");

        // Récupération de base : les lemmes des thématiques, tout simplement
        GraphLemmaItem[] lThe = themas.getItems(ThemaGraph.BASE_UNIQUEMENT);
        for(int i=0; i<lThe.length; i++){
            statut = "Enrichissement thématique de base ("+(i+1)+"/"+lThe.length+");";
            tax.addThematique(lThe[i]);
        }

        // Si enrichissement par spécialisation, récupération des lemmes des
        // sous-thématiques
        if(isEnrichThemaSpe()){
            lThe = themas.getItems(ThemaGraph.ENRICH_UNIQUEMENT);
            for (int i = 0; i < lThe.length; i++) {
                statut = "Enrichissement thématique par spécialisation ("+(i+1)+"/"+lThe.length+");";
                tax.addInThematique(lThe[i]);
            }
        }

        if(isEnrichThemaSem()){
            for(int i=0; i<tax.getThematiques().length; i++){
                statut = "Enrichissement thématique par champ sémantique ("+(i+1)+"/"+tax.getThematiques().length+")";
                System.out.println(Horloge.getHMS()+"   Thématique "+(i+1)+"/"+tax.getThematiques().length+" : "+tax.getThematiques()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<tax.getThematiques()[i].getAllLemmes().length; j++){
                    LemmePondere lemme = tax.getThematiques()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+tax.getThematiques()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getChampSemantique(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.CHAMP_SEMANTIQUE);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    tax.getThematiques()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }

        if(isEnrichThemaCat()){
            for(int i=0; i<tax.getThematiques().length; i++){
                statut = "Enrichissement thématique par catégorisation ("+(i+1)+"/"+tax.getThematiques().length+")";
                System.out.println(Horloge.getHMS()+"   Thématique "+(i+1)+"/"+tax.getThematiques().length+" : "+tax.getThematiques()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<tax.getThematiques()[i].getAllLemmes().length; j++){
                    LemmePondere lemme = tax.getThematiques()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+tax.getThematiques()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getGeneralisationWiki(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.CATEGORISATION);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    tax.getThematiques()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }

        // Si enrichissement par généralisation, récupération des lemmes génériques de DBPedia.
        if(isEnrichThemaGen()){
            for(int i=0; i<tax.getThematiques().length; i++){
                statut = "Enrichissement thématique par généralisation ("+(i+1)+"/"+tax.getThematiques().length+")";
                System.out.println(Horloge.getHMS()+"   Thématique "+(i+1)+"/"+tax.getThematiques().length+" : "+tax.getThematiques()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<tax.getThematiques()[i].getAllLemmes().length; j++){
                    //if(j==0) System.out.println("    Test de l'enrichissement par généralisation");
                    LemmePondere lemme = tax.getThematiques()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+tax.getThematiques()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    //if(j==0) System.out.println("    Premier lemme : "+lemme.getLemme()+" ("+lemme.getCategorie()+")");
                    // enrichissement sémantique de 'lemme'.
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getGeneralisationOnto(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        //if(j==0) System.out.println("    Nombre de parents dans l'ontologie : "+enrichissement.length);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.GENERALISATION);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    tax.getThematiques()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }
    }

    private void creeNuages() throws NoBaseException {
        //******* CONSTITUTION DES NUAGES **************************************
        System.out.println(Horloge.getHMS()+" Constitution des nuages.");
        if(cor != null){
            creeNuageCorpus();
        }
        if(tax != null){
            creeNuageTaxonomie();
        }

        // * Pondération
        statut = "Pondération en cours";
        System.out.println(Horloge.getHMS()+" Pondération");

        if(!poidsOrigines.isEmpty()){
            Vector<Operation> v = new Vector<Operation>(poidsOrigines.keySet());
            for(int i=0; i<v.size(); i++){
                cor.pondereOrigine(v.elementAt(i), poidsOrigines.get(v.elementAt(i)).doubleValue());
                tax.pondereOrigine(v.elementAt(i), poidsOrigines.get(v.elementAt(i)).doubleValue());
            }
        }
        cor.pondereAllCategories(poidsCategoriesNonSpecifie);
        tax.pondereAllCategories(poidsCategoriesNonSpecifie);
        if(!poidsCategories.isEmpty()){
            Vector<String> v = new Vector<String>(poidsCategories.keySet());
            for(int i=0; i<v.size(); i++){
                cor.pondereCategorie(v.elementAt(i), poidsCategories.get(v.elementAt(i)).doubleValue());
                tax.pondereCategorie(v.elementAt(i), poidsCategories.get(v.elementAt(i)).doubleValue());
            }
        }
        
        if (normalisationTaille) {
            tax.normaliseParTaille();
        }
    }

    private String[] creeVecteur(NuagePondere[] nuage){
        Set<String> listeLemmes = new HashSet<String>();
        for(int i=0; i<nuage.length; i++){
            for(int j=0; j<nuage[i].getAllLemmes().length; j++){
                LemmePondere aAjouter = nuage[i].getAllLemmes()[j];
                if(aAjouter.getPoidsCategorie()>0) {
                    listeLemmes.add(aAjouter.getLemme()+":"+aAjouter.getCategorie());
                }
            }
        }
        String[] vecteur = new String[listeLemmes.size()];
        listeLemmes.toArray(vecteur);
        return vecteur;
    }

    private boolean demarree = false;
    private boolean termine = true;
    private long depart;
    private long fin;

    @Override
    public void run() {
        demarree = true;
        System.out.println(Horloge.getHMS()+titre+" : expérience lancée.");
        if(!termine){
            System.out.println(Horloge.getHMS()+" Erreur : Précédente expérience en cours.");
            return;
        }
        termine = false;
        // Initialisation
        depart = System.currentTimeMillis();
        if(corpus==null || themas==null){
            // Pas de thématique ou pas d'article : pas de confrontation
            termine = true;
            fin = System.currentTimeMillis();
            System.out.println(Horloge.getHMS()+" Erreur : Pas de thématique ou pas d'article.");
            return;
        }
        if(!isEnrichThemaBase() || !isEnrichArticleBase()){
            // Pas d'enrichissement de base : on arrête tout
            // (pour l'instant du moins)
            termine = true;
            fin = System.currentTimeMillis();
            System.out.println(Horloge.getHMS()+" Erreur : Pas d'enrichissement de base.");
            return;
        }
        for(int i=0; i<themaTrouvees.length; i++){
            themaTrouvees[i] = -1;
        }

        //******* CONSTITUTION DES NUAGES **************************************
        try {
            creeNuages();
        } catch(NoBaseException nbe){

        }

        //******* Confrontation des nuages *************************************

        statut = "Confrontation en cours";
        NuagePondere[] articles = cor.getArticles();
        NuagePondere[] thematiques = tax.getThematiques();
        String[] vecteur = creeVecteur(thematiques);
        double[][] matriceThematiques = new double[thematiques.length][vecteur.length];
        double[][] matriceArticles = new double[articles.length][vecteur.length];

        System.out.println(Horloge.getHMS()+" Confrontation : "+articles.length+" articles, "+thematiques.length+" thématiques.");

        // Fabrication de la matrice des vecteurs des thématiques
        for(int i=0; i<thematiques.length; i++){
            statut = "Vectorisation en cours (thématique "+(i+1)+"/"+articles.length+")";
            LemmePondere[] lemmes = thematiques[i].getAllLemmes();
            for(int j=0; j<lemmes.length; j++){
                if(lemmes[j].getPoidsCategorie()>0){
                    String lemme = lemmes[j].getLemme()+":"+lemmes[j].getCategorie();
                    int index = 0;
                    for(int a=0; a<vecteur.length; a++){
                        if(vecteur[a].equals(lemme)){
                            index = a;
                            break;
                        }
                    }
                    matriceThematiques[i][index] = lemmes[j].getPoidsOrigine()*
                            lemmes[j].getPoidsCategorie()*
                            lemmes[j].getNbOccurences();
                }
            }
        }

        // Fabrication de la matrice des vecteurs des articles
        for(int i=0; i<articles.length; i++){
            statut = "Vectorisation en cours (article "+(i+1)+"/"+articles.length+")";
            LemmePondere[] lemmes = articles[i].getAllLemmes();
            for(int j=0; j<lemmes.length; j++){
                if(lemmes[j].getPoidsCategorie()>0){
                    String lemme = lemmes[j].getLemme()+":"+lemmes[j].getCategorie();
                    int index = 0;
                    for(int a=0; a<vecteur.length; a++){
                        if(vecteur[a].equals(lemme)){
                            index = a;
                            break;
                        }
                    }
                    matriceArticles[i][index] = lemmes[j].getPoidsOrigine()*
                            lemmes[j].getPoidsCategorie()*
                            lemmes[j].getNbOccurences();
                }
            }
        }

        // Comparaison des matrices
        for(int i=0; i<articles.length; i++){
            statut = "Confrontation en cours (article "+(i+1)+"/"+articles.length+")";
            double[] scores = new double[thematiques.length];
            for(int j=0; j<thematiques.length; j++){
                scores[j] = 0.0;
                if(algoCalcul==VECTEUR_PRODUIT_SCALAIRE) {
                    for(int a=0; a<vecteur.length; a++){
                        scores[j] += matriceArticles[i][a] * matriceThematiques[j][a];
                    }
                } else if(algoCalcul==VECTEUR_DISTANCE_EUCLIDIENNE) {
                    for(int a=0; a<vecteur.length; a++){
                        // Soustraction car il faut minimiser la distance
                        scores[j] -= (matriceArticles[i][a] - matriceThematiques[j][a]) *
                                (matriceArticles[i][a] - matriceThematiques[j][a]);
                    }
                    scores[j] = -1*Math.sqrt(-1*scores[j]);
                } else if(algoCalcul==VECTEUR_DISTANCE_DE_MANHATTAN) {
                    for(int a=0; a<vecteur.length; a++){
                        // Soustraction car il faut minimiser la distance
                        scores[j] -= Math.abs(matriceArticles[i][a] - matriceThematiques[j][a]);
                    }
                }
            }
            // On choisit la première thématique ayant le plus gros score
            double max = 0;
            for(int j=0; j<thematiques.length; j++){
                if(scores[j]>max){
                    max = scores[j];
                }
            }
            if (max > 0) {
                for (int j = 0; j < thematiques.length; j++) {
                    if (scores[j] == max) {
                        themaTrouvees[i] = thematiques[j].getId();
                    }
                }
            } else {
                themaTrouvees[i] = 0;
            }
            boolean v = themaTrouvees[i]==corpus.getItems()[i].getIdThematique();
            System.out.println(Horloge.getHMS()+"  "+(v?"O":"X")
                    +" Article "+(i+1)+
                    " | Thématique trouvée : "+ themaTrouvees[i]+
                    " (théoriquement "+corpus.getItems()[i].getIdThematique()+")");
        }
        
        /*for(int i=0; i<articles.length; i++){
            statut = "Confrontation en cours (article "+(i+1)+"/"+articles.length+")";
            double[] scores = new double[thematiques.length];
            for(int j=0; j<thematiques.length; j++){
                scores[j] = 0.0;

                Vector<LemmePondere> lemmesArticle = articles[i].getAllLemmesAsVector();
                Vector<LemmePondere> lemmesThema = thematiques[j].getAllLemmesAsVector();

                // On fait l'intersection : les lemmes qui ne sont pas
                // des deux côtés n'ont aucun intérêt
                //lemmesArticle.retainAll(lemmesThema);
                //lemmesThema.retainAll(lemmesArticle); // (marche pas)

                // Comparaison
                for(int a = 0; a<lemmesArticle.size(); a++){
                    for(int b=0; b<lemmesThema.size(); b++){
                        if(lemmesArticle.elementAt(a).getPoidsOrigine() *
                           lemmesThema.elementAt(b).getPoidsOrigine() *
                           lemmesArticle.elementAt(a).getPoidsCategorie() *
                           lemmesThema.elementAt(b).getPoidsCategorie() > 0) {
                            if (lemmesArticle.elementAt(a).getLemme().equals(lemmesThema.elementAt(b).getLemme())) {
                                // Rajout du produit des poids (la similarité est égale à 1)
                                // Attention : la quotité respective des lemmes
                                // (le nombre de fois où ils apparaissent) n'est pas
                                // prise en compte !
                                scores[j] +=
                                        lemmesArticle.elementAt(a).getPoidsOrigine() *
                                        lemmesThema.elementAt(b).getPoidsOrigine() *
                                        lemmesArticle.elementAt(a).getPoidsCategorie() *
                                        lemmesThema.elementAt(b).getPoidsCategorie();
                            }
                        }
                    }
                }
            }
            // On choisit la première thématique ayant le plus gros score
            double max = 0;
            for(int j=0; j<thematiques.length; j++){
                if(scores[j]>max){
                    max = scores[j];
                }
            }
            if (max > 0) {
                for (int j = 0; j < thematiques.length; j++) {
                    if (scores[j] == max) {
                        themaTrouvees[i] = thematiques[j].getId();
                    }
                }
            } else {
                themaTrouvees[i] = 0;
            }
            boolean v = themaTrouvees[i]==corpus.getItems()[i].getIdThematique();
            System.out.println(Horloge.getHMS()+"  "+(v?"O":"X")
                    +" Article "+(i+1)+
                    " | Thématique trouvée : "+ themaTrouvees[i]+
                    " (théoriquement "+corpus.getItems()[i].getIdThematique()+")");
        }*/

        // Terminé !
        System.out.println(Horloge.getHMS()+" Expérience terminée : pertinence "+((int)(100*getPertinence()))+"%.");

        termine = true;
        fin = System.currentTimeMillis();
    }
    
    public int getSecondesExecution(){
        if(termine){
            return (int)((fin-depart)/1000);
        } else {
            return (int)((System.currentTimeMillis()-depart)/1000);
        }
    }

    public String getStatut(){
        if(!demarree){
            return "Lancer l'expérience";
        } else if(termine){
            return "Expérience terminée";
        } else {
            return statut;
        }
    }
    
    public boolean isTermine(){
        return termine;
    }

    public double getPertinence(){
        if(corpus == null){
            return 0.0;
        }
        int corrects = 0;
        int verifiables = 0;
        CorpusGraphLemmaItem[] lCorpus = corpus.getItems();
        for(int i=0; i<themaTrouvees.length; i++){
            if(themaTrouvees[i]!=-1 && lCorpus[i].getIdThematique() != 0){
                verifiables++;
                if(themaTrouvees[i] == lCorpus[i].getIdThematique()){
                    corrects++;
                } else {
                }
            }
        }
        if(verifiables>0){
            return 1.0*corrects/verifiables;
        } else {
            return 0;
        }
    }

    public boolean isPertinent(int index) {
        if(themaTrouvees==null) {
            return false;
        }
        if (index < 0 || index >= themaTrouvees.length) {
            return false;
        }
        CorpusGraphLemmaItem[] lCorpus = corpus.getItems();
        boolean res = false;
        if (themaTrouvees[index] != -1 && lCorpus[index].getIdThematique() != 0) {
            if (themaTrouvees[index] == lCorpus[index].getIdThematique()) {
                res = true;
            } else {
            }
        }
        return res;
    }


}
